package main

import (
	"bytes"
	"encoding/binary"
	"fmt"
	"os"
	"syscall"
	"time"

)

const (
	I2CBus        = "/dev/i2c-1"
	AHT10I2CAddr  = 0x38
	BMP180I2CAddr = 0x77
)

type CalibrationData struct {
	AC1, AC2, AC3, B1, B2, MB, MC, MD int16
	AC4, AC5, AC6                      uint16
}

// Opens the I2C bus
func openI2CBus() (*os.File, error) {
	return os.OpenFile(I2CBus, os.O_RDWR, 0644)
}

// Sets the I2C slave device address
func setI2CDevice(file *os.File, addr int) error {
	if _, _, errno := syscall.Syscall(syscall.SYS_IOCTL, file.Fd(), uintptr(0x0703), uintptr(addr)); errno != 0 {
		return fmt.Errorf("error setting I2C address: %v", errno)
	}
	return nil
}

// Reads a specified number of bytes from a register
func readRegister(file *os.File, reg byte, length int) ([]byte, error) {
	if _, err := file.Write([]byte{reg}); err != nil {
		return nil, fmt.Errorf("error writing to I2C: %v", err)
	}
	buf := make([]byte, length)
	if _, err := file.Read(buf); err != nil {
		return nil, fmt.Errorf("error reading from I2C: %v", err)
	}
	return buf, nil
}

// Reads temperature and humidity from the AHT10 sensor
func readAHT10(file *os.File) (float64, float64, error) {
	cmd := []byte{0xAC, 0x33, 0x00}
	if _, err := file.Write(cmd); err != nil {
		return 0, 0, err
	}
	time.Sleep(80 * time.Millisecond) // Wait for measurement

	data := make([]byte, 6)
	if _, err := file.Read(data); err != nil {
		return 0, 0, err
	}

	// Check if data is valid
	if len(data) < 6 || data[0]&0x80 != 0 {
		return 0, 0, fmt.Errorf("sensor not ready or invalid data")
	}

	// Convert raw data to humidity and temperature values
	rawHumidity := (uint32(data[1])<<16 | uint32(data[2])<<8 | uint32(data[3])) >> 4
	rawTemperature := (uint32(data[3]&0x0F)<<16 | uint32(data[4])<<8 | uint32(data[5]))

	humidity := float64(rawHumidity) * 100.0 / 1048576.0
	temperature := float64(rawTemperature)*200.0/1048576.0 - 50

	return temperature, humidity, nil
}

// Reads calibration data from the BMP180 sensor
func readCalibrationData(file *os.File) (*CalibrationData, error) {
	data, err := readRegister(file, 0xAA, 22)
	if err != nil {
		return nil, err
	}
	var calib CalibrationData
	if err := binary.Read(bytes.NewReader(data), binary.BigEndian, &calib); err != nil {
		return nil, fmt.Errorf("error parsing calibration data: %v", err)
	}
	return &calib, nil
}

// Writes sensor data to a file with a timestamp
func writeToFile(filename string, value float64, unit string) error {
	file, err := os.OpenFile(filename, os.O_WRONLY|os.O_CREATE|os.O_APPEND, 0644)
	if err != nil {
		return err
	}
	defer file.Close()
	timestamp := time.Now().Format("2006-01-02 15:04:05.000")
	_, err = fmt.Fprintf(file, "%.2f\t%s\t%s\n", value, unit, timestamp)
	return err
}

func main() {
	file, err := openI2CBus()
	if err != nil {
		fmt.Println(err)
		return
	}
	defer file.Close()

	// Set I2C device address for AHT10
	if err := setI2CDevice(file, AHT10I2CAddr); err != nil {
		fmt.Println("Failed to acquire AHT10:", err)
		return
	}

	// Continuous sensor reading loop
	for {
		tempAHT, humAHT, err := readAHT10(file)
		if err == nil {
			fmt.Printf("AHT10 Temp: %.2f C, Humidity: %.2f %%\n", tempAHT, humAHT)
			_ = writeToFile("temperature.tsv", tempAHT, "C")
			_ = writeToFile("humidity.tsv", humAHT, "RH")
		} else {
			fmt.Println("Error reading AHT10:", err)
		}
		time.Sleep(1 * time.Second)
	}
}
