#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <dirent.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include "commands.h"
#include "utils.h"

void _unix() {
    time_t result = time(NULL);
    printf("Unix Timestamp: %lld\n", (long long)result);
}

void ls() {
    DIR* dir;
    struct dirent* entry;
    char** files = NULL;
    size_t count = 0;

    dir = opendir(".");
    if (dir == NULL) {
        printf("Failed to open directory\n");
        return;
    }

    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") != 0 && strcmp(entry->d_name, "..") != 0) {
            files = realloc(files, (count + 1) * sizeof(char*));
            if (files == NULL) {
                perror("realloc");
                closedir(dir);
                return;
            }
            files[count] = strdup(entry->d_name);
            if (files[count] == NULL) {
                perror("strdup");
                closedir(dir);
                return;
            }
            count++;
        }
    }
    closedir(dir);

    qsort(files, count, sizeof(char*), compareStrings);

    for (size_t i = 0; i < count; i++) {
        printf("%s%s\n", files[i], getTypeSuffix(files[i]));
        usleep(5000);
        free(files[i]);
    }
    free(files);
}

void date() {
    struct timespec ts;
    if (clock_gettime(CLOCK_REALTIME, &ts) == -1) {
        perror("clock_gettime");
        return;
    }

    struct tm timeinfo;
    if (localtime_r(&ts.tv_sec, &timeinfo) == NULL) {
        perror("localtime_r");
        return;
    }

    char buffer[100];
    if (strftime(buffer, sizeof(buffer), "%a %b %d %H:%M:%S %Z %Y", &timeinfo) == 0) {
        fprintf(stderr, "strftime returned 0\n");
        return;
    }

    printf("%s\n", buffer);
}

void whoami() {
    const char* username = getenv("USER");
    if (username) {
        printf("%s\n", username);
    } else {
        perror("Failed to get username.");
    }
}

void cat(const char* filename) {
    FILE* file = fopen(filename, "r");
    if (!file) {
        perror("cat");
        return;
    }
    char buffer[256];
    while (fgets(buffer, sizeof(buffer), file)) {
        printf("%s", buffer);
    }
    fclose(file);
}

void nl(const char* filename) {
    FILE* file = fopen(filename, "r");
    if (!file) {
        perror("nl");
        return;
    }
    char buffer[256];
    int line = 1;
    while (fgets(buffer, sizeof(buffer), file)) {
        printf("%6d\t%s", line++, buffer);
    }
    fclose(file);
}

void cd(const char* path) {
    if (path == NULL) {
        const char* home = getenv("HOME");
        if (home == NULL) {
            fprintf(stderr, "cd: HOME not set\n");
            return;
        }
        path = home;
    }
    if (chdir(path) != 0) {
        perror("cd");
    }
}

void execute_program(const char* path, char* const argv[]) {
    pid_t pid = fork();
    if (pid == -1) {
        perror("fork");
        return;
    }
    if (pid == 0) {
        execvp(path, argv);
        perror("execvp");
        exit(EXIT_FAILURE);
    } else {
        int status;
        if (waitpid(pid, &status, 0) == -1) {
            perror("waitpid");
        }
    }
}

void help() {
    printf("Available commands:\n");
    printf("exit\n");
    printf("test\n");
    printf("clear\n");
    printf("unix\n");
    printf("ls\n");
    printf("date\n");
    printf("whoami\n");
    printf("cat\n");
    printf("nl\n");
    printf("cd\n");
    printf("help\n");
    printf("exec\n");
}
